/**
* \file: MFiServerPlatformHooks.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPlay
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <authentication.h>
#include <ipodauth.h>
#include "AirPlayHeaders.h"
#include "Common.h"
#include "mutex"

using namespace adit::carplay;

extern "C" {
// symbol replacement hooks for Apple communication plug-in core defined in this file:
CARPLAY_EXPORTED OSStatus MFiPlatform_Initialize(void);
CARPLAY_EXPORTED void     MFiPlatform_Finalize(void);
CARPLAY_EXPORTED OSStatus MFiPlatform_CopyCertificate(uint8_t** outCertificatePtr,
        size_t* outCertificateLen);
CARPLAY_EXPORTED OSStatus MFiPlatform_CreateSignature(const void* inDigestPtr,
        size_t inDigestLen, uint8_t** outSignaturePtr, size_t* outSignatureLen);
}

// structure to cache certificate
struct _Certificate
{
    uint8_t* data;
    size_t   len;
    std::mutex mtx;

    _Certificate() : data(nullptr), len(0) {};
    ~_Certificate() { Release(); };
    void Release();
    void Create();
    bool Check();
};
static _Certificate _certificate;

// ========== MFiPlatform hooks ==========

OSStatus MFiPlatform_Initialize(void)
{
    if (_certificate.data != nullptr)
    {
        LOG_WARN((dipo, "MFi certificate was already initialized"));
        _certificate.Release();
    }

    _certificate.mtx.lock();
    _certificate.Create();
    _certificate.mtx.unlock();
    return kNoErr;
}

void MFiPlatform_Finalize(void)
{
    _certificate.Release();
}

OSStatus MFiPlatform_CopyCertificate(uint8_t** outCertificatePtr, size_t* outCertificateLen)
{
    if (outCertificatePtr == nullptr || outCertificateLen == nullptr)
    {
        LOG_ERROR((dipo, "MFiPlatform_CopyCertificate: invalid arguments!"));
        return kParamErr;
    }

    *outCertificatePtr = nullptr;

    _certificate.mtx.lock();

    if (!_certificate.Check())
    {
        _certificate.mtx.unlock();
        // error logged
        return kUnknownErr;
    }

    _certificate.mtx.unlock();

    // communication plug-in core expects the memory to be allocated with malloc
    uint8_t* ptr = static_cast<uint8_t*>(malloc(_certificate.len));
    dipo_exit_on_null(ptr);

    memcpy(ptr, _certificate.data, _certificate.len);

    *outCertificatePtr = ptr;
    *outCertificateLen = _certificate.len;
    return kNoErr;
}

OSStatus MFiPlatform_CreateSignature(const void* inDigestPtr, size_t inDigestLen,
        uint8_t** outSignaturePtr, size_t* outSignatureLen)
{
    if (inDigestPtr == nullptr ||  inDigestLen != IPOD_AUTH_CP_CHALLENGE_DATA_SIZE ||
            outSignaturePtr == nullptr || outSignatureLen == nullptr)
    {
        LOG_ERROR((dipo, "MFiPlatform_CreateSignature: invalid arguments!"));
        return kParamErr;
    }

    *outSignaturePtr = nullptr;
    *outSignatureLen = 0;

    U16 len = 0;
    // communication plug-in core expects the memory to be allocated with malloc
    uint8_t* ptr = static_cast<uint8_t*>(malloc(IPOD_AUTH_CP_SIGN_DATA_SIZE));
    int ret = AuthenticationGetSignatureData(static_cast<const U8*>(inDigestPtr), &len, ptr);

    if (ret == IPOD_AUTH_OK && len > 0)
    {
        *outSignaturePtr = ptr;
        *outSignatureLen = len;
    }
    else
    {
        LOG_ERROR((dipo, "AuthenticationGetSignatureData failed!"));
        free(ptr);
        return kUnknownErr;
    }

    return kNoErr;
}

// ========== _Certificate methods ==========

void _Certificate::Create()
{
    // cache certificate
    U16 len = 0;
    uint8_t* ptr = new uint8_t[IPOD_AUTH_CP_MAX_CERTLENGTH];
    dipo_exit_on_null(ptr);

    AuthenticationGetCertificate(&len, ptr);
    if (len == 0)
    {
        LOG_ERROR((dipo, "AuthenticationGetCertificate failed!"));
        delete [] ptr;
    }
    else
    {
        this->data = ptr;
        this->len = len;
    }
}

bool _Certificate::Check()
{
    bool err = false;

    // check if already created
    if (data != nullptr && len > 0)
        err = true;
    else
    {
        LOG_WARN((dipo, "MFi certificate is not initialized yet!!"));
        Create();
        if(data == nullptr || len == 0)
            LOG_ERROR((dipo, "Error in MFi certificate initialization. Please check gpio settings!!!"));
        else
            err = true;
    }

    return err;
}

void _Certificate::Release()
{
    dipo_safe_delete_array(data);
    len = 0;
}
